/*
 *  Copyright (C) 2006-2019  Ronald Blankendaal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
package org.dbgl.model.conf;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.List;
import org.apache.commons.io.FileUtils;
import org.dbgl.model.conf.mount.Mount;
import org.dbgl.service.ITextService;


public class Configuration extends Settings {

	private static final String AUTOEXEC_SECTION_NAME = "autoexec";

	private Autoexec autoexec_;

	public Configuration() {
		super();
		autoexec_ = new Autoexec();
	}

	public Configuration(Configuration configuration) {
		this();
		sections_ = new SectionsCollection(configuration.getSections());
		autoexec_ = new Autoexec(configuration.getAutoexec());
	}

	public Autoexec getAutoexec() {
		return autoexec_;
	}

	public void setAutoexec(Autoexec autoexec) {
		autoexec_ = autoexec;
	}

	public String reloadWithAutoexec(ITextService text, List<Mount> dbMounts) throws IOException {
		sections_ = new SectionsCollection();
		autoexec_ = new Autoexec();

		File file = fileLocation_.getCanonicalFile();
		if (file == null || !file.isFile() || !file.canRead())
			throw new IOException(text.get("general.error.openfile", new Object[] {String.valueOf(file)}));

		try (FileReader reader = new FileReader(file)) {
			return loadDataWithAutoexec(text, reader, file, dbMounts);
		}
	}

	public String loadDataWithAutoexec(ITextService text, String data, File file, List<Mount> dbMounts) throws IOException {
		try (StringReader reader = new StringReader(data)) {
			return loadDataWithAutoexec(text, reader, file, dbMounts);
		}
	}

	private String loadDataWithAutoexec(ITextService text, Reader reader, File file, List<Mount> dbMounts) throws IOException {
		StringBuffer warningsLog = new StringBuffer();

		List<String> autoexecLines = new ArrayList<>();
		warningsLog.append(loadData(text, reader, file, AUTOEXEC_SECTION_NAME, autoexecLines));
		if (autoexecLines.size() > 0)
			warningsLog.append(autoexec_.load(autoexecLines, dbMounts));

		return warningsLog.toString();
	}

	public void save(boolean prepareOnly, List<Mount> combinedMountingPoints) throws IOException {
		File file = fileLocation_.getCanonicalFile();
		FileUtils.forceMkdirParent(file);
		try (BufferedWriter writer = new BufferedWriter(new FileWriter(file))) {
			writeData(writer, prepareOnly, combinedMountingPoints);
		}
	}

	private void writeData(BufferedWriter writer, boolean prepareOnly, List<Mount> combinedMountingPoints) throws IOException {
		writer.write(sections_.toString());
		writer.write(autoexec_.toString(prepareOnly, combinedMountingPoints));
	}

	public void removeUnequalValuesIn(Configuration configuration) {
		super.removeUnequalValuesIn(configuration);
		autoexec_.removeUnequalValuesIn(configuration.autoexec_);
	}

	public void removeUnnecessaryMounts(Configuration configuration) {
		autoexec_.removeUnnecessaryMounts(configuration.autoexec_);
	}

	public String toString(List<Mount> combinedMountingPoints) {
		StringWriter result = new StringWriter();
		try (BufferedWriter writer = new BufferedWriter(result)) {
			writeData(writer, false, combinedMountingPoints);
		} catch (IOException e) {
			new RuntimeException(e);
		}
		return result.toString();
	}
}
