/*
 *  Copyright (C) 2006-2020  Ronald Blankendaal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
package org.dbgl.gui.abstractdialog;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StackLayout;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Shell;


public abstract class WizardDialog<T> extends SizeControlledDialog<T> {

	private final StackLayout wizardLayout_;

	protected Composite contents_;
	protected Button backButton_;
	protected Button nextButton_;
	protected Button cancelButton_;
	protected int step_;

	public WizardDialog(Shell parent, String dialogName) {
		super(parent, dialogName);

		wizardLayout_ = new StackLayout();
		step_ = 0;
	}

	@Override
	protected void onShellInit() {
		super.onShellInit();

		shell_.setLayout(new GridLayout());

		contents_ = createComposite(shell_, new GridData(SWT.FILL, SWT.FILL, true, true));
		contents_.setLayout(wizardLayout_);

		Composite buttons_ = createInnerComposite(shell_, new GridData(SWT.FILL, SWT.FILL, true, false), 2);
		Composite backNextButtons_ = createInnerComposite(buttons_, new GridData(SWT.BEGINNING, SWT.FILL, false, false), 2);

		backButton_ = createButton(backNextButtons_, text_.get("button.back"), new SelectionAdapter() {
			public void widgetSelected(SelectionEvent event) {
				goBack();
			}
		});

		nextButton_ = createButton(backNextButtons_, text_.get("button.next"), new SelectionAdapter() {
			public void widgetSelected(SelectionEvent event) {
				goForward();
			}
		});
		shell_.setDefaultButton(nextButton_);

		GridLayout gridLayoutOtherButtons = new GridLayout(2, true);
		gridLayoutOtherButtons.marginWidth = 80;
		gridLayoutOtherButtons.marginHeight = 0;
		Composite otherButtons_ = createComposite(buttons_, new GridData(SWT.BEGINNING, SWT.FILL, false, false), gridLayoutOtherButtons);

		cancelButton_ = createButton(otherButtons_, text_.get("button.cancel"), closeShellAdapter);

		setLayoutDataButtons(backButton_, nextButton_, cancelButton_);
	}

	protected void addStep(Group group) {
		group.setParent(contents_);
	}

	@Override
	protected void onShellOpened() {
		activateCurrentStep();
	}

	protected int stepSize(int step, boolean forward) {
		return 1;
	}

	protected void goBack() {
		step_ -= stepSize(step_, false);
		activateCurrentStep();
	}

	protected void goForward() {
		if (onNext(step_)) {
			if (step_ >= (totalSteps() - 1)) {
				shell_.close();
			} else {
				step_ += stepSize(step_, true);
				activateCurrentStep();
			}
		}
	}

	protected abstract boolean onNext(int step);

	protected void activateCurrentStep() {
		if (step_ < contents_.getChildren().length) {
			wizardLayout_.topControl = contents_.getChildren()[step_];
			contents_.layout();
		}

		backButton_.setEnabled(step_ > 0);
		nextButton_.setText(step_ < (totalSteps() - 1) ? text_.get("button.next"): text_.get("button.finish"));

		if (step_ > 0)
			wizardLayout_.topControl.setFocus();
	}

	protected int totalSteps() {
		return contents_.getChildren().length;
	}
}
