/*
 *  Copyright (C) 2006-2020  Ronald Blankendaal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
package org.dbgl.gui.dialog;

import org.apache.commons.lang3.StringUtils;
import org.dbgl.gui.GeneralPurposeDialogs;
import org.dbgl.gui.abstractdialog.SizeControlledTabbedDialog;
import org.dbgl.gui.controls.BrowseButton;
import org.dbgl.gui.controls.BrowseButton.BrowseType;
import org.dbgl.gui.controls.BrowseButton.CanonicalType;
import org.dbgl.model.NativeCommand;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;


public class EditNativeCommandDialog extends SizeControlledTabbedDialog<NativeCommand> {

	private final NativeCommand nativeCommand_;

	public EditNativeCommandDialog(Shell parent, NativeCommand cmd) {
		super(parent, "nativecommanddialog");
		nativeCommand_ = cmd;
	}

	@Override
	protected String getDialogTitle() {
		return text_.get(((nativeCommand_ == null) ? "dialog.nativecommand.title.add": "dialog.nativecommand.title.edit"));
	}

	@Override
	protected void onShellCreated() {
		Composite composite = createTabWithComposite(text_.get("dialog.nativecommand.tab.info"), 3);

		Text command = createLabelAndText(composite, text_.get("dialog.nativecommand.command"));
		command.setFocus();
		BrowseButton cmdBrowseButton = new BrowseButton(composite);

		Text parameters = createLabelAndText(composite, text_.get("dialog.nativecommand.parameters"), 2);

		Text cwd = createLabelAndText(composite, text_.get("dialog.nativecommand.cwd"));
		BrowseButton cwdBrowseButton = new BrowseButton(composite);

		cmdBrowseButton.connect(shell_, command, cwd, BrowseType.FILE, CanonicalType.NATIVE_EXE, false, null);
		cwdBrowseButton.connect(shell_, cwd, null, BrowseType.DIR, CanonicalType.NATIVE_EXE, false, null);

		Button waitFor = createLabelAndCheckButton(composite, text_.get("dialog.nativecommand.waitfor"), 2);

		createOkCancelButtons(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent event) {
				if (!isValid(command, cwd)) {
					return;
				}
				int orderNr = (nativeCommand_ == null) ? -1: nativeCommand_.getOrderNr();
				result_ = new NativeCommand(command.getText(), parameters.getText(), cwd.getText(), waitFor.getSelection(), orderNr);
				shell_.close();
			}
		});

		if (nativeCommand_ != null) {
			command.setText(nativeCommand_.getCommand().getPath());
			parameters.setText(nativeCommand_.getParameters());
			cwd.setText(nativeCommand_.getCwd().getPath());
			waitFor.setSelection(nativeCommand_.isWaitFor());
		}
	}

	private boolean isValid(Text command, Text cwd) {
		GeneralPurposeDialogs.initErrorDialog();
		if (StringUtils.isBlank(command.getText())) {
			GeneralPurposeDialogs.addError(text_.get("dialog.nativecommand.required.command"), command);
		} else if (StringUtils.isBlank(cwd.getText())) {
			GeneralPurposeDialogs.addError(text_.get("dialog.nativecommand.required.cwd"), cwd);
		}
		return !GeneralPurposeDialogs.displayErrorDialog(shell_);
	}
}
