/*
 *  Copyright (C) 2006-2024  Ronald Blankendaal
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
package org.dbgl.tool;

import java.io.File;
import java.io.IOException;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.dbgl.model.aggregate.DosboxVersion;
import org.dbgl.model.aggregate.Profile;
import org.dbgl.model.repository.DosboxVersionRepository;
import org.dbgl.model.repository.ProfileRepository;
import org.dbgl.service.DatabaseService;
import org.dbgl.service.DatabaseService.Transaction;
import org.dbgl.service.SettingsService;


public class RenumberDatabase {

	private static final String TMP_FILE_POSTFIX = "__TMP__";

	private static final String CREATE_NEW_TABLES_QRY = 
			  "CREATE MEMORY TABLE NEW_DEVELOPERS(ID INTEGER GENERATED BY DEFAULT AS IDENTITY(START WITH 0) NOT NULL PRIMARY KEY,ORG_ID INTEGER,NAME VARCHAR(256) NOT NULL);"
			+ "CREATE MEMORY TABLE NEW_PUBLISHERS(ID INTEGER GENERATED BY DEFAULT AS IDENTITY(START WITH 0) NOT NULL PRIMARY KEY,ORG_ID INTEGER,NAME VARCHAR(256) NOT NULL);"
			+ "CREATE MEMORY TABLE NEW_GENRES(ID INTEGER GENERATED BY DEFAULT AS IDENTITY(START WITH 0) NOT NULL PRIMARY KEY,ORG_ID INTEGER,NAME VARCHAR(256) NOT NULL);"
			+ "CREATE MEMORY TABLE NEW_PUBLYEARS(ID INTEGER GENERATED BY DEFAULT AS IDENTITY(START WITH 0) NOT NULL PRIMARY KEY,ORG_ID INTEGER,YEAR VARCHAR(256) NOT NULL);"
			+ "CREATE MEMORY TABLE NEW_STATUS(ID INTEGER GENERATED BY DEFAULT AS IDENTITY(START WITH 0) NOT NULL PRIMARY KEY,ORG_ID INTEGER,STAT VARCHAR(256) NOT NULL);"
			+ "CREATE MEMORY TABLE NEW_CUSTOM1(ID INTEGER GENERATED BY DEFAULT AS IDENTITY(START WITH 0) NOT NULL PRIMARY KEY,ORG_ID INTEGER,VALUE VARCHAR(256) NOT NULL);"
			+ "CREATE MEMORY TABLE NEW_CUSTOM2(ID INTEGER GENERATED BY DEFAULT AS IDENTITY(START WITH 0) NOT NULL PRIMARY KEY,ORG_ID INTEGER,VALUE VARCHAR(256) NOT NULL);"
			+ "CREATE MEMORY TABLE NEW_CUSTOM3(ID INTEGER GENERATED BY DEFAULT AS IDENTITY(START WITH 0) NOT NULL PRIMARY KEY,ORG_ID INTEGER,VALUE VARCHAR(256) NOT NULL);"
			+ "CREATE MEMORY TABLE NEW_CUSTOM4(ID INTEGER GENERATED BY DEFAULT AS IDENTITY(START WITH 0) NOT NULL PRIMARY KEY,ORG_ID INTEGER,VALUE VARCHAR(256) NOT NULL);"
			+ "CREATE MEMORY TABLE NEW_DOSBOXVERSIONS(ID INTEGER GENERATED BY DEFAULT AS IDENTITY(START WITH 0) NOT NULL PRIMARY KEY,ORG_ID INTEGER,TITLE VARCHAR(256) NOT NULL,PATH VARCHAR(256) NOT NULL,EXEFILE VARCHAR(256) NOT NULL,CONFFILE VARCHAR(256) NOT NULL,MULTICONF BOOLEAN,ISDEFAULT BOOLEAN,PARAMETERS VARCHAR(256) DEFAULT '',VERSION VARCHAR(256) NOT NULL,FAMILY VARCHAR(256) NOT NULL,USINGCURSES BOOLEAN,STATS_CREATED TIMESTAMP(0) DEFAULT CURRENT_TIMESTAMP NOT NULL,STATS_LASTMODIFY TIMESTAMP(0),STATS_LASTRUN TIMESTAMP(0),STATS_RUNS INTEGER DEFAULT 0 NOT NULL, DYNAMIC_OPTIONS OTHER DEFAULT NULL);"
			+ "CREATE MEMORY TABLE NEW_TEMPLATES(ID INTEGER GENERATED BY DEFAULT AS IDENTITY(START WITH 0) NOT NULL PRIMARY KEY,ORG_ID INTEGER,TITLE VARCHAR(256) NOT NULL,DBVERSION_ID INTEGER,ISDEFAULT BOOLEAN,STATS_CREATED TIMESTAMP(0) DEFAULT CURRENT_TIMESTAMP NOT NULL,STATS_LASTMODIFY TIMESTAMP(0),STATS_LASTRUN TIMESTAMP(0),STATS_RUNS INTEGER DEFAULT 0 NOT NULL,CONSTRAINT SYS_FK_185 FOREIGN KEY(DBVERSION_ID) REFERENCES NEW_DOSBOXVERSIONS(ID));"
			+ "CREATE MEMORY TABLE NEW_GAMES(ID INTEGER GENERATED BY DEFAULT AS IDENTITY(START WITH 0) NOT NULL PRIMARY KEY,ORG_ID INTEGER,"
			+ " TITLE VARCHAR(256) NOT NULL,DEV_ID INTEGER,PUBL_ID INTEGER,GENRE_ID INTEGER,YEAR_ID INTEGER,STAT_ID INTEGER,NOTES LONGVARCHAR,FAVORITE BOOLEAN,"
			+ " SETUP VARCHAR(256),SETUP_PARAMS VARCHAR(256),ALT1 VARCHAR(256) DEFAULT '',ALT1_PARAMS VARCHAR(256) DEFAULT '',ALT2 VARCHAR(256) DEFAULT '',ALT2_PARAMS VARCHAR(256) DEFAULT '',"
			+ " CONFFILE VARCHAR(256),CAPTURES VARCHAR(256),DBVERSION_ID INTEGER, LINK1 VARCHAR(256),LINK2 VARCHAR(256),LINK3 VARCHAR(256) DEFAULT '',LINK4 VARCHAR(256) DEFAULT '',"
			+ " LINK5 VARCHAR(256) DEFAULT '',LINK6 VARCHAR(256) DEFAULT '',LINK7 VARCHAR(256) DEFAULT '',LINK8 VARCHAR(256) DEFAULT '',"
			+ " LINK1_TITLE VARCHAR(256) DEFAULT '',LINK2_TITLE VARCHAR(256) DEFAULT '',LINK3_TITLE VARCHAR(256) DEFAULT '',LINK4_TITLE VARCHAR(256) DEFAULT '',"
			+ " LINK5_TITLE VARCHAR(256) DEFAULT '',LINK6_TITLE VARCHAR(256) DEFAULT '',LINK7_TITLE VARCHAR(256) DEFAULT '',LINK8_TITLE VARCHAR(256) DEFAULT '',"
			+ " CUST1_ID INTEGER DEFAULT 0,CUST2_ID INTEGER DEFAULT 0,CUST3_ID INTEGER DEFAULT 0,CUST4_ID INTEGER DEFAULT 0,"
			+ " CUSTOM5 VARCHAR(256) DEFAULT '',CUSTOM6 VARCHAR(256) DEFAULT '',CUSTOM7 VARCHAR(256) DEFAULT '',CUSTOM8 VARCHAR(256) DEFAULT '',"
			+ " CUSTOM9 INTEGER DEFAULT 0,CUSTOM10 INTEGER DEFAULT 0, STATS_CREATED TIMESTAMP(0) DEFAULT CURRENT_TIMESTAMP NOT NULL,STATS_LASTMODIFY TIMESTAMP(0),"
			+ " STATS_LASTRUN TIMESTAMP(0),STATS_RUNS INTEGER DEFAULT 0 NOT NULL,STATS_LASTSETUP TIMESTAMP(0),STATS_SETUPS INTEGER DEFAULT 0 NOT NULL,"
			+ " CUSTOM11 VARCHAR(256) DEFAULT '',CUSTOM12 VARCHAR(256) DEFAULT '',CUSTOM13 VARCHAR(256) DEFAULT '',CUSTOM14 VARCHAR(256) DEFAULT '', MOUNT_IDX INTEGER DEFAULT 0,"
			+ " CONSTRAINT SYS_FK_165 FOREIGN KEY(DEV_ID) REFERENCES NEW_DEVELOPERS(ID),CONSTRAINT SYS_FK_166 FOREIGN KEY(PUBL_ID) REFERENCES NEW_PUBLISHERS(ID),"
			+ " CONSTRAINT SYS_FK_167 FOREIGN KEY(GENRE_ID) REFERENCES NEW_GENRES(ID),CONSTRAINT SYS_FK_168 FOREIGN KEY(YEAR_ID) REFERENCES NEW_PUBLYEARS(ID),"
			+ " CONSTRAINT SYS_FK_169 FOREIGN KEY(DBVERSION_ID) REFERENCES NEW_DOSBOXVERSIONS(ID),CONSTRAINT SYS_FK_170 FOREIGN KEY(STAT_ID) REFERENCES NEW_STATUS(ID));"
			+ "CREATE MEMORY TABLE NEW_NATIVECOMMANDS(ID INTEGER GENERATED BY DEFAULT AS IDENTITY(START WITH 0) NOT NULL PRIMARY KEY,ORG_ID INTEGER,COMMAND VARCHAR(256) NOT NULL,PARAMETERS VARCHAR(256) NOT NULL,CWD VARCHAR(256) NOT NULL,WAITFOR BOOLEAN,ORDERNR INTEGER,GAME_ID INTEGER,TEMPLATE_ID INTEGER, CONSTRAINT SYS_FK_180 FOREIGN KEY(GAME_ID) REFERENCES NEW_GAMES(ID),CONSTRAINT SYS_FK_181 FOREIGN KEY(TEMPLATE_ID) REFERENCES NEW_TEMPLATES(ID));"
			+ "CREATE MEMORY TABLE NEW_LOG (ID INTEGER GENERATED BY DEFAULT AS IDENTITY(START WITH 0) NOT NULL PRIMARY KEY,ORG_ID INTEGER,TIME TIMESTAMP(0) DEFAULT CURRENT_TIMESTAMP NOT NULL,EVENT TINYINT NOT NULL, ENTITY_TYPE TINYINT NOT NULL,ENTITY_ID INT NOT NULL, ENTITY_TITLE VARCHAR(256) NOT NULL);";
	private static final String INSERT_INTO_NEW_TABLES_QRY = 
			  "INSERT INTO NEW_DEVELOPERS (SELECT NULL,ID,NAME FROM DEVELOPERS ORDER BY NAME);"
			+ "INSERT INTO NEW_PUBLISHERS (SELECT NULL,ID,NAME FROM PUBLISHERS ORDER BY NAME);"
			+ "INSERT INTO NEW_GENRES (SELECT NULL,ID,NAME FROM GENRES ORDER BY NAME);"
			+ "INSERT INTO NEW_PUBLYEARS (SELECT NULL,ID,YEAR FROM PUBLYEARS ORDER BY YEAR);"
			+ "INSERT INTO NEW_STATUS (SELECT NULL,ID,STAT FROM STATUS ORDER BY STAT);"
			+ "INSERT INTO NEW_CUSTOM1 (SELECT NULL,ID,VALUE FROM CUSTOM1 ORDER BY VALUE);"
			+ "INSERT INTO NEW_CUSTOM2 (SELECT NULL,ID,VALUE FROM CUSTOM2 ORDER BY VALUE);"
			+ "INSERT INTO NEW_CUSTOM3 (SELECT NULL,ID,VALUE FROM CUSTOM3 ORDER BY VALUE);"
			+ "INSERT INTO NEW_CUSTOM4 (SELECT NULL,ID,VALUE FROM CUSTOM4 ORDER BY VALUE);"
			+ "INSERT INTO NEW_DOSBOXVERSIONS (SELECT NULL,ID,TITLE,PATH,EXEFILE,CONFFILE,MULTICONF,ISDEFAULT,PARAMETERS,VERSION,FAMILY,USINGCURSES,STATS_CREATED,STATS_LASTMODIFY,STATS_LASTRUN,STATS_RUNS,DYNAMIC_OPTIONS FROM DOSBOXVERSIONS ORDER BY TITLE);"
			+ "INSERT INTO NEW_TEMPLATES ("
			+ "  SELECT NULL,ID,TITLE,DBV.ID,ISDEFAULT,STATS_CREATED,STATS_LASTMODIFY,STATS_LASTRUN,STATS_RUNS"
			+ "  FROM TEMPLATES TMPL, NEW_DOSBOXVERSIONS DBV WHERE TMPL.DBVERSION_ID = DBV.ORG_ID ORDER BY TITLE);"
			+ "INSERT INTO NEW_GAMES ("
			+ "  SELECT NULL,ID,TITLE,DEV.ID,PUBL.ID,GENRE.ID,YR.ID,STAT.ID,NOTES,FAVORITE,SETUP,SETUP_PARAMS,"
			+ "  ALT1,ALT1_PARAMS,ALT2,ALT2_PARAMS,CONFFILE,CAPTURES,DBV.ID,"
			+ "  LINK1,LINK2,LINK3,LINK4,LINK5,LINK6,LINK7,LINK8,LINK1_TITLE,LINK2_TITLE,LINK3_TITLE,LINK4_TITLE,LINK5_TITLE,LINK6_TITLE,"
			+ "  LINK7_TITLE,LINK8_TITLE,CUST1.ID,CUST2.ID,CUST3.ID,CUST4.ID,CUSTOM5,CUSTOM6,CUSTOM7,CUSTOM8,CUSTOM9,CUSTOM10,"
			+ "  STATS_CREATED,STATS_LASTMODIFY,STATS_LASTRUN,STATS_RUNS,STATS_LASTSETUP,STATS_SETUPS,"
			+ "  CUSTOM11,CUSTOM12,CUSTOM13,CUSTOM14,MOUNT_IDX"
			+ "  FROM GAMES GAM,NEW_DEVELOPERS DEV,NEW_PUBLISHERS PUBL,NEW_GENRES GENRE,NEW_PUBLYEARS YR,NEW_STATUS STAT,"
			+ "  NEW_DOSBOXVERSIONS DBV,NEW_CUSTOM1 CUST1,NEW_CUSTOM2 CUST2,NEW_CUSTOM3 CUST3,NEW_CUSTOM4 CUST4"
			+ "  WHERE GAM.DBVERSION_ID = DBV.ORG_ID AND GAM.DEV_ID = DEV.ORG_ID AND GAM.PUBL_ID = PUBL.ORG_ID"
			+ "  AND GAM.GENRE_ID = GENRE.ORG_ID AND GAM.YEAR_ID = YR.ORG_ID AND GAM.STAT_ID = STAT.ORG_ID"
			+ "  AND GAM.CUST1_ID = CUST1.ORG_ID AND GAM.CUST2_ID = CUST2.ORG_ID"
			+ "  AND GAM.CUST3_ID = CUST3.ORG_ID AND GAM.CUST4_ID = CUST4.ORG_ID ORDER BY TITLE);"
			+ "INSERT INTO NEW_NATIVECOMMANDS ("
			+ "  SELECT NULL,ID,COMMAND,PARAMETERS,CWD,WAITFOR,ORDERNR,GAM.ID,TMPL.ID FROM NATIVECOMMANDS NAT, NEW_GAMES GAM,NEW_TEMPLATES TMPL"
			+ "  WHERE ( (NAT.GAME_ID = GAM.ORG_ID AND NAT.TEMPLATE_ID IS NULL) OR (NAT.TEMPLATE_ID = TMPL.ORG_ID AND NAT.GAME_ID IS NULL) ) ORDER BY COMMAND);"
			+ "INSERT INTO NEW_LOG ("
			+ "  SELECT * FROM (SELECT NULL,ID,TIME,EVENT,ENTITY_TYPE,GAM.ID,ENTITY_TITLE FROM LOG LOG, NEW_GAMES GAM WHERE LOG.ENTITY_ID = GAM.ORG_ID AND LOG.ENTITY_TYPE = 0 UNION"
			+ "	 SELECT NULL,ID,TIME,EVENT,ENTITY_TYPE,DBV.ID,ENTITY_TITLE FROM LOG LOG, NEW_DOSBOXVERSIONS DBV WHERE LOG.ENTITY_ID = DBV.ORG_ID AND LOG.ENTITY_TYPE = 1 UNION"
			+ "	 SELECT NULL,ID,TIME,EVENT,ENTITY_TYPE,TMPL.ID,ENTITY_TITLE FROM LOG LOG, NEW_TEMPLATES TMPL WHERE LOG.ENTITY_ID = TMPL.ORG_ID AND LOG.ENTITY_TYPE = 2 UNION"
			+ "	 SELECT NULL,ID,TIME,EVENT,ENTITY_TYPE,FIL.ID,ENTITY_TITLE FROM LOG LOG, FILTERS FIL WHERE LOG.ENTITY_ID = FIL.ID AND LOG.ENTITY_TYPE = 3) ORDER BY TIME);";
	private static final String DROP_OLD_IDS_IN_NEW_TABLES_QRY =
			  "ALTER TABLE NEW_DEVELOPERS DROP COLUMN ORG_ID; ALTER TABLE NEW_PUBLISHERS DROP COLUMN ORG_ID; ALTER TABLE NEW_GENRES DROP COLUMN ORG_ID;"
			+ "ALTER TABLE NEW_PUBLYEARS DROP COLUMN ORG_ID; ALTER TABLE NEW_STATUS DROP COLUMN ORG_ID; ALTER TABLE NEW_CUSTOM1 DROP COLUMN ORG_ID;"
			+ "ALTER TABLE NEW_CUSTOM2 DROP COLUMN ORG_ID; ALTER TABLE NEW_CUSTOM3 DROP COLUMN ORG_ID; ALTER TABLE NEW_CUSTOM4 DROP COLUMN ORG_ID;"
			+ "ALTER TABLE NEW_DOSBOXVERSIONS DROP COLUMN ORG_ID; ALTER TABLE NEW_TEMPLATES DROP COLUMN ORG_ID; ALTER TABLE NEW_GAMES DROP COLUMN ORG_ID;"
			+ "ALTER TABLE NEW_NATIVECOMMANDS DROP COLUMN ORG_ID; ALTER TABLE NEW_LOG DROP COLUMN ORG_ID;";
	private static final String DROP_ORG_TABLES_QRY = 
			  "DROP TABLE LOG; DROP TABLE NATIVECOMMANDS; DROP TABLE GAMES; DROP TABLE TEMPLATES; DROP TABLE DOSBOXVERSIONS; DROP TABLE CUSTOM4;"
			+ "DROP TABLE CUSTOM3; DROP TABLE CUSTOM2; DROP TABLE CUSTOM1; DROP TABLE STATUS; DROP TABLE PUBLYEARS; DROP TABLE GENRES;"
			+ "DROP TABLE PUBLISHERS; DROP TABLE DEVELOPERS;";
	private static final String RENAME_NEW_TABLES_QRY = 
			  "ALTER TABLE NEW_DEVELOPERS RENAME TO DEVELOPERS; ALTER TABLE NEW_PUBLISHERS RENAME TO PUBLISHERS; ALTER TABLE NEW_GENRES RENAME TO GENRES;"
			+ "ALTER TABLE NEW_PUBLYEARS RENAME TO PUBLYEARS; ALTER TABLE NEW_STATUS RENAME TO STATUS; ALTER TABLE NEW_CUSTOM1 RENAME TO CUSTOM1;"
			+ "ALTER TABLE NEW_CUSTOM2 RENAME TO CUSTOM2; ALTER TABLE NEW_CUSTOM3 RENAME TO CUSTOM3; ALTER TABLE NEW_CUSTOM4 RENAME TO CUSTOM4;"
			+ "ALTER TABLE NEW_DOSBOXVERSIONS RENAME TO DOSBOXVERSIONS; ALTER TABLE NEW_TEMPLATES RENAME TO TEMPLATES; ALTER TABLE NEW_GAMES RENAME TO GAMES;"
			+ "ALTER TABLE NEW_NATIVECOMMANDS RENAME TO NATIVECOMMANDS; ALTER TABLE NEW_LOG RENAME TO LOG;";
	
	public static void main(String[] args) {
		System.out.println("Renumber database (v0.99)");
		System.out.println();

		DatabaseService dbService = DatabaseService.getInstance();

		try {
			try (Connection con = dbService.getConnection()) {
				runBatch(dbService, con, CREATE_NEW_TABLES_QRY, "Create new tables");
				runBatch(dbService, con, INSERT_INTO_NEW_TABLES_QRY, "Copy data into new tables");
				runBatch(dbService, con, DROP_OLD_IDS_IN_NEW_TABLES_QRY, "Drop old IDs");
				runBatch(dbService, con, DROP_ORG_TABLES_QRY, "Drop old tables");
				runBatch(dbService, con, RENAME_NEW_TABLES_QRY, "Rename new tables");
			}
			updateProfileCapturesAndConfs();
		} catch (Exception e) {
			e.printStackTrace();
		}

		System.out.println("Finished.");
		try {
			dbService.shutdown();
		} catch (SQLException e) {
			// nothing we can do
		}
	}

	private static void runBatch(DatabaseService dbService, Connection con, String qry, String msg) throws SQLException {
		System.out.print(msg + "... ");
		try (Transaction transaction = dbService.startTransaction();
				Statement stmt = con.createStatement()) {
			for (String s: qry.split(";"))
				stmt.addBatch(s);
			stmt.executeBatch();
			transaction.commit();
		}
		System.out.println("done");
	}

	private static void updateProfileCapturesAndConfs() throws SQLException, IOException {
		List<DosboxVersion> dbversionsList = new DosboxVersionRepository().listAll();
		List<Profile> profiles = new ProfileRepository().
				list(" ORDER BY GAM.ID", StringUtils.EMPTY, dbversionsList);
		
		for (Profile prof: profiles) {
			System.out.print("Profile " + prof.getTitle() + " (" + prof.getId() + ") ...");
			
			String warnings = prof.resetAndLoadConfiguration();
			if (!warnings.isEmpty())
				System.out.println(warnings);
			
			File caps = prof.getCapturesFile();
			int oldId = Integer.parseInt(caps.getName());
			
			if ((oldId < profiles.size()) && (oldId != prof.getId())) {
				File canCaps = prof.getCanonicalCaptures();
				prof.setCapturesFileLocation(canCaps + TMP_FILE_POSTFIX);
				prof.setCapturesInConfig();
				if (canCaps.renameTo(prof.getCanonicalCaptures())) {
					System.out.print(" " + caps + "=>" + prof.getCapturesFile());
				} else {
					System.err.print("rename " + caps + "=>" + prof.getCapturesFile() + " failed");
				}
				
				if (SettingsService.getInstance().getIntValue("profiledefaults", "conffile") == 0) {
					File conf = prof.getConfigurationFile();
					File canConf = prof.getConfigurationCanonicalFile();
					prof.setConfigurationFileLocation(canConf + TMP_FILE_POSTFIX);
					if (canConf.renameTo(prof.getConfigurationCanonicalFile())) {
						System.out.print(" " + conf + "=>" + prof.getConfigurationFile());
					} else {
						System.out.print("rename " + conf + "=>" + prof.getConfigurationFile() + " failed");
					}
				}
			}
			
			System.out.println();
		}
		
		for (Profile prof: profiles) {
			System.out.print("Profile " + prof.getTitle() + " (" + prof.getId() + ") ...");
			
			File caps = prof.getCapturesFile();
			int oldId = Integer.parseInt(StringUtils.removeEnd(caps.getName(), TMP_FILE_POSTFIX));
			
			if (oldId != prof.getId()) {
				File canCaps = prof.getCanonicalCaptures();
				prof.setCapturesById();
				prof.setCapturesInConfig();
				if (canCaps.renameTo(prof.getCanonicalCaptures())) {
					System.out.print(" " + caps + "=>" + prof.getCapturesFile());
				} else {
					System.err.print("rename " + caps + "=>" + prof.getCapturesFile() + " failed");
				}
				
				if (SettingsService.getInstance().getIntValue("profiledefaults", "conffile") == 0) {
					File conf = prof.getConfigurationFile();
					File canConf = prof.getConfigurationCanonicalFile();
					prof.setConfigurationFileLocationByIdentifiers();
					if (canConf.renameTo(prof.getConfigurationCanonicalFile())) {
						System.out.print(" " + conf + "=>" + prof.getConfigurationFile());
					} else {
						System.out.print("rename " + conf + "=>" + prof.getConfigurationFile() + " failed");
					}
				}
				
				new ProfileRepository().update(prof);
			}
			
			System.out.println();
		}
	}
}
